<?php
/**
 * Plugin Name: HTML Import
 * Plugin URI: https://example.com/
 * Description: HTMLファイルからWordPressページを生成するプラグインです。
 * Version: 1.3.0
 * Author: Enakat
 * Author URI: https://enakat.com/
 * License: GPL2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: html-import
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * プラグイン初期化
 */
function html_import_init() {
    // 必要に応じて初期化処理を追加
}
add_action( 'init', 'html_import_init' );

/**
 * 管理メニューを追加
 */
function html_import_admin_menu() {
    add_menu_page(
        __( 'HTML Import', 'html-import' ),
        __( 'HTML Import', 'html-import' ),
        'manage_options',
        'html-import',
        'html_import_render_admin_page',
        'dashicons-upload',
        30
    );
}
add_action( 'admin_menu', 'html_import_admin_menu' );

/**
 * 管理画面に通知を表示します。
 */
function html_import_admin_notices() {
    if ( ! empty( $_GET['html_import_notice'] ) ) {
        $notice_type = sanitize_key( $_GET['html_import_notice'] );
        $message = '';
        $class = '';

        switch ( $notice_type ) {
            case 'success_page_import_summary':
                $imported = isset( $_GET['imported'] ) ? intval( $_GET['imported'] ) : 0;
                $skipped = isset( $_GET['skipped'] ) ? intval( $_GET['skipped'] ) : 0;
                $failed = isset( $_GET['failed'] ) ? intval( $_GET['failed'] ) : 0;
                $message = sprintf(
                    __( 'ページインポート処理が完了しました。追加：%d件、スキップ：%d件、失敗：%d件', 'html-import' ),
                    $imported,
                    $skipped,
                    $failed
                );
                $class = ( $failed > 0 || ( $imported === 0 && $skipped > 0 && $failed === 0 ) ) ? 'notice-warning' : 'notice-success';
                break;
            case 'success_index_page_set':
                $message = __( 'インデックスページ(LP)を設定しました。', 'html-import' );
                $class = 'notice-success';
                break;
            case 'error_index_page_set':
                $message = __( 'インデックスページ(LP)の設定に失敗しました。', 'html-import' );
                $class = 'notice-error';
                break;
            case 'error_upload':
                $message = __( 'ファイルのアップロードに失敗しました。ファイルが選択されていないか、形式が正しくありません。', 'html-import' );
                $class = 'notice-error';
                break;
            default:
                return;
        }

        if ( ! empty( $message ) ) {
            printf( '<div class="notice %s is-dismissible"><p>%s</p></div>', esc_attr( $class ), esc_html( $message ) );
        }
    }
}
add_action( 'admin_notices', 'html_import_admin_notices' );

/**
 * 管理画面のメインページをレンダリング
 */
function html_import_render_admin_page() {
    ?>
    <div class="wrap">
        <h1><?php echo esc_html( get_admin_page_title() ); ?></h1>
        <div class="tab-content" style="padding-top: 20px;">
            <?php html_import_render_page_import_tab(); ?>
        </div>
    </div>
    <?php
}

/**
 * ページ追加タブをレンダリングします。
 */
function html_import_render_page_import_tab() {
    ?>
    <h3><?php esc_html_e( 'ページ追加', 'html-import' ); ?></h3>
    <p><?php esc_html_e( 'HTMLファイルを複数選択して、新しいページを一括で追加します。', 'html-import' ); ?></p>
    <form action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>" method="post" enctype="multipart/form-data">
        <input type="hidden" name="action" value="html_import_import_page">
        <?php wp_nonce_field( 'html_import_page_import' ); ?>
        <table class="form-table">
            <tbody>
                <tr>
                    <th scope="row"><label for="html_files"><?php esc_html_e( 'HTMLファイル', 'html-import' ); ?></label></th>
                    <td><input type="file" name="html_files[]" id="html_files" accept=".html,.htm" required multiple></td>
                </tr>
                <tr>
                    <th scope="row"><?php esc_html_e( '追加する種類', 'html-import' ); ?></th>
                    <td>
                        <fieldset>
                            <label><input type="radio" name="post_type" value="post" checked> <?php esc_html_e( '投稿ページとして追加する', 'html-import' ); ?></label><br>
                            <label><input type="radio" name="post_type" value="page"> <?php esc_html_e( '固定ページとして追加する', 'html-import' ); ?></label>
                        </fieldset>
                    </td>
                </tr>
            </tbody>
        </table>
        <?php submit_button( __( 'ページを追加', 'html-import' ) ); ?>
    </form>

    <?php
    // 既存の固定ページ一覧を表示
    $pages = get_pages( array(
        'sort_column' => 'post_date',
        'sort_order' => 'DESC',
        'number' => 100,
    ) );

    if ( ! empty( $pages ) ) {
        $show_on_front = get_option( 'show_on_front' );
        $page_on_front = get_option( 'page_on_front' );
        ?>
        <hr style="margin: 30px 0;">
        <h3><?php esc_html_e( '既存の固定ページ一覧', 'html-import' ); ?></h3>
        <p><?php esc_html_e( 'インデックスページ(LP)を選択すると、そのページがサイトのトップページとして設定されます。', 'html-import' ); ?></p>
        
        <form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>">
            <input type="hidden" name="action" value="html_import_set_index_page">
            <?php wp_nonce_field( 'html_import_set_index_page' ); ?>
            
            <table class="wp-list-table widefat fixed striped">
                <thead>
                    <tr>
                        <th style="width: 50px;"><?php esc_html_e( '選択', 'html-import' ); ?></th>
                        <th><?php esc_html_e( 'ページタイトル', 'html-import' ); ?></th>
                        <th><?php esc_html_e( 'ステータス', 'html-import' ); ?></th>
                        <th><?php esc_html_e( '作成日', 'html-import' ); ?></th>
                        <th><?php esc_html_e( '操作', 'html-import' ); ?></th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ( $pages as $page ) : 
                        $is_current_front = ( $show_on_front === 'page' && $page_on_front == $page->ID );
                        $status_labels = array(
                            'publish' => __( '公開済み', 'html-import' ),
                            'draft' => __( '下書き', 'html-import' ),
                            'pending' => __( '承認待ち', 'html-import' ),
                            'private' => __( '非公開', 'html-import' ),
                        );
                        $status = isset( $status_labels[ $page->post_status ] ) ? $status_labels[ $page->post_status ] : $page->post_status;
                    ?>
                        <tr <?php if ( $is_current_front ) echo 'style="background-color: #e7f5e7;"'; ?> >
                            <td>
                                <input type="radio" name="index_page_id" value="<?php echo esc_attr( $page->ID ); ?>" <?php checked( $is_current_front ); ?>> 
                            </td>
                            <td>
                                <strong><?php echo esc_html( $page->post_title ); ?></strong>
                                <?php if ( $is_current_front ) : ?>
                                    <span style="color: #46b450; font-weight: bold;">● <?php esc_html_e( '現在のトップページ', 'html-import' ); ?></span>
                                <?php endif; ?>
                            </td>
                            <td><?php echo esc_html( $status ); ?></td>
                            <td><?php echo esc_html( get_the_date( 'Y/m/d H:i', $page->ID ) ); ?></td>
                            <td>
                                <a href="<?php echo esc_url( get_edit_post_link( $page->ID ) ); ?>" class="button button-small"><?php esc_html_e( '編集', 'html-import' ); ?></a>
                                <a href="<?php echo esc_url( get_permalink( $page->ID ) ); ?>" class="button button-small" target="_blank"><?php esc_html_e( '表示', 'html-import' ); ?></a>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
            
            <p class="submit">
                <button type="submit" class="button button-primary"><?php esc_html_e( '選択したページをトップページに設定', 'html-import' ); ?></button>
                <?php if ( $show_on_front === 'page' && $page_on_front > 0 ) : ?>
                    <button type="submit" name="remove_index_page" value="1" class="button" onclick="return confirm('<?php esc_attr_e( 'トップページの設定を解除してもよろしいですか?', 'html-import' ); ?>');">
                        <?php esc_html_e( 'トップページ設定を解除', 'html-import' ); ?>
                    </button>
                <?php endif; ?>
            </p>
        </form>

        <div class="notice notice-info inline" style="margin-top: 20px;">
            <p>
                <strong><?php esc_html_e( '設定について:', 'html-import' ); ?></strong><br>
                <?php esc_html_e( '• Twenty Twentyシリーズ: トップページが正常に表示されます。', 'html-import' ); ?><br>
                <?php esc_html_e( '• Cocoon: テーマ設定により、フロントページタイプが「投稿ページ」になっている場合は、「固定ページ」に変更してください。', 'html-import' ); ?><br>
                <?php esc_html_e( '• その他のテーマ: テーマによっては追加の設定が必要な場合があります。', 'html-import' ); ?>
            </p>
        </div>
        <?php
    }
}

/**
 * HTMLをWordPressブロックに変換します。
 */
function html_import_convert_html_to_blocks( $html_content ) {
    $dom = new DOMDocument();
    
    libxml_use_internal_errors( true );
    $dom->loadHTML( '<?xml encoding="UTF-8">' . $html_content, LIBXML_HTML_NOIMPLIED | LIBXML_HTML_NODEFDTD );
    libxml_clear_errors();

    $xpath = new DOMXPath( $dom );
    
    $main_content = html_import_extract_main_content( $xpath, $dom );
    
    if ( empty( $main_content ) ) {
        return '';
    }

    $blocks = array();
    
    foreach ( $main_content as $element ) {
        $block = html_import_element_to_block( $element, $dom );
        if ( ! empty( $block ) ) {
            $blocks[] = $block;
        }
    }

    return implode( "\n\n", $blocks );
}

/**
 * <header>と<footer>の間のコンテンツを抽出します。
 */
function html_import_extract_main_content( $xpath, $dom ) {
    $main_elements = array();
    
    $headers = $xpath->query( '//header' );
    $footers = $xpath->query( '//footer' );
    
    if ( $headers->length > 0 && $footers->length > 0 ) {
        $header = $headers->item(0);
        $footer = $footers->item(0);
        
        $current = $header->nextSibling;
        while ( $current && $current !== $footer ) {
            if ( $current->nodeType === XML_ELEMENT_NODE ) {
                $main_elements[] = $current;
            }
            $current = $current->nextSibling;
        }
    } else {
        $body = $xpath->query( '//body' )->item(0);
        if ( $body ) {
            foreach ( $body->childNodes as $child ) {
                if ( $child->nodeType === XML_ELEMENT_NODE && 
                     $child->nodeName !== 'header' && 
                     $child->nodeName !== 'footer' ) {
                    $main_elements[] = $child;
                }
            }
        }
    }
    
    return $main_elements;
}

/**
 * DOM要素をWordPressブロックに変換します。
 */
function html_import_element_to_block( $element, $dom ) {
    $tag_name = strtolower( $element->nodeName );
    
    if ( in_array( $tag_name, array( 'div', 'section', 'article', 'aside', 'main' ) ) ) {
        return html_import_create_group_block( $element, $dom );
    }
    
    if ( preg_match( '/^h[1-6]$/', $tag_name ) ) {
        $level = substr( $tag_name, 1 );
        $content = html_import_get_inner_html( $element, $dom );
        return '<!-- wp:heading {"level":' . $level . '} -->' . "\n" . 
               '<' . $tag_name . '>' . $content . '</' . $tag_name . '>' . "\n" . 
               '<!-- /wp:heading -->';
    }
    
    if ( $tag_name === 'p' ) {
        $content = html_import_get_inner_html( $element, $dom );
        if ( trim( strip_tags( $content ) ) === '' ) {
            return '';
        }
        return '<!-- wp:paragraph -->' . "\n" . 
               '<p>' . $content . '</p>' . "\n" . 
               '<!-- /wp:paragraph -->';
    }
    
    if ( $tag_name === 'img' ) {
        $src = $element->getAttribute( 'src' );
        $alt = $element->getAttribute( 'alt' );
        return '<!-- wp:image -->' . "\n" . 
               '<figure class="wp-block-image"><img src="' . esc_url( $src ) . '" alt="' . esc_attr( $alt ) . '"/></figure>' . "\n" . 
               '<!-- /wp:image -->';
    }
    
    if ( in_array( $tag_name, array( 'ul', 'ol' ) ) ) {
        $content = html_import_get_inner_html( $element, $dom );
        return '<!-- wp:list -->' . "\n" . 
               '<' . $tag_name . '>' . $content . '</' . $tag_name . '>' . "\n" . 
               '<!-- /wp:list -->';
    }
    
    $content = $dom->saveHTML( $element );
    if ( empty( trim( strip_tags( $content ) ) ) ) {
        return '';
    }
    
    return '<!-- wp:html -->' . "\n" . 
           $content . "\n" . 
           '<!-- /wp:html -->';
}

/**
 * グループブロックを作成します。
 */
function html_import_create_group_block( $element, $dom ) {
    $inner_blocks = array();
    
    foreach ( $element->childNodes as $child ) {
        if ( $child->nodeType === XML_ELEMENT_NODE ) {
            $block = html_import_element_to_block( $child, $dom );
            if ( ! empty( $block ) ) {
                $inner_blocks[] = $block;
            }
        }
    }
    
    if ( empty( $inner_blocks ) ) {
        $content = $dom->saveHTML( $element );
        return '<!-- wp:html -->' . "\n" . 
               $content . "\n" . 
               '<!-- /wp:html -->';
    }
    
    $class_attr = $element->getAttribute( 'class' );
    $id_attr = $element->getAttribute( 'id' );
    
    $attrs = array();
    if ( ! empty( $class_attr ) ) {
        $attrs[] = '"className":"' . esc_attr( $class_attr ) . '"';
    }
    if ( ! empty( $id_attr ) ) {
        $attrs[] = '"anchor":"' . esc_attr( $id_attr ) . '"';
    }
    
    $attrs_string = ! empty( $attrs ) ? ' {' . implode( ',', $attrs ) . '}' : '';
    
    return '<!-- wp:group' . $attrs_string . ' -->' . "\n" . 
           '<div class="wp-block-group">' . "\n" . 
           implode( "\n\n", $inner_blocks ) . "\n" . 
           '</div>' . "\n" . 
           '<!-- /wp:group -->';
}

/**
 * 要素の内部HTMLを取得します。
 */
function html_import_get_inner_html( $element, $dom ) {
    $inner_html = '';
    foreach ( $element->childNodes as $child ) {
        $inner_html .= $dom->saveHTML( $child );
    }
    return $inner_html;
}

/**
 * ページインポート処理を実行します。
 */
function html_import_handle_page_import_form() {
    check_admin_referer( 'html_import_page_import' );

    if ( ! current_user_can( 'manage_options' ) ) {
        wp_die( __( '権限がありません。', 'html-import' ) );
    }

    $imported = 0;
    $skipped = 0;
    $failed = 0;

    if ( empty( $_FILES['html_files']['name'][0] ) ) {
        wp_redirect( add_query_arg( array(
            'page' => 'html-import',
            'html_import_notice' => 'error_upload',
        ), admin_url( 'admin.php' ) ) );
        exit;
    }

    $post_type = isset( $_POST['post_type'] ) && $_POST['post_type'] === 'page' ? 'page' : 'post';
    $files = $_FILES['html_files'];

    for ( $i = 0; $i < count( $files['name'] ); $i++ ) {
        if ( $files['error'][$i] !== UPLOAD_ERR_OK ) {
            $failed++;
            continue;
        }

        $file_name = sanitize_file_name( $files['name'][$i] );
        $file_path = $files['tmp_name'][$i];
        
        $file_ext = strtolower( pathinfo( $file_name, PATHINFO_EXTENSION ) );
        if ( ! in_array( $file_ext, array( 'html', 'htm' ) ) ) {
            $failed++;
            continue;
        }

        $html_content = file_get_contents( $file_path );
        if ( $html_content === false ) {
            $failed++;
            continue;
        }

        $page_title = pathinfo( $file_name, PATHINFO_FILENAME );
        $page_title = str_replace( array( '-', '_' ), ' ', $page_title );
        $page_title = ucwords( $page_title );

        $existing_post = get_page_by_title( $page_title, OBJECT, $post_type );
        if ( $existing_post ) {
            $skipped++;
            continue;
        }

        $block_content = html_import_convert_html_to_blocks( $html_content );

        if ( empty( $block_content ) ) {
            $failed++;
            continue;
        }

        $post_data = array(
            'post_title'   => $page_title,
            'post_content' => $block_content,
            'post_status'  => 'draft',
            'post_type'    => $post_type,
            'post_author'  => get_current_user_id(),
        );

        $post_id = wp_insert_post( $post_data );

        if ( is_wp_error( $post_id ) || $post_id === 0 ) {
            $failed++;
        } else {
            $imported++;
        }
    }

    wp_redirect( add_query_arg( array(
        'page' => 'html-import',
        'html_import_notice' => 'success_page_import_summary',
        'imported' => $imported,
        'skipped' => $skipped,
        'failed' => $failed
    ), admin_url( 'admin.php' ) ) );
    exit;
}
add_action( 'admin_post_html_import_import_page', 'html_import_handle_page_import_form' );

/**
 * インデックスページ設定処理を実行します。
 */
function html_import_handle_set_index_page() {
    check_admin_referer( 'html_import_set_index_page' );

    if ( ! current_user_can( 'manage_options' ) ) {
        wp_die( __( '権限がありません。', 'html-import' ) );
    }

    if ( isset( $_POST['remove_index_page'] ) && $_POST['remove_index_page'] === '1' ) {
        update_option( 'show_on_front', 'posts' );
        update_option( 'page_on_front', 0 );
        delete_option( 'html_import_index_page_id' );
        
        wp_redirect( add_query_arg( array(
            'page' => 'html-import',
            'html_import_notice' => 'success_index_page_set',
        ), admin_url( 'admin.php' ) ) );
        exit;
    }

    $page_id = isset( $_POST['index_page_id'] ) ? intval( $_POST['index_page_id'] ) : 0;

    if ( $page_id <= 0 ) {
        wp_redirect( add_query_arg( array(
            'page' => 'html-import',
            'html_import_notice' => 'error_index_page_set',
        ), admin_url( 'admin.php' ) ) );
        exit;
    }

    $page = get_post( $page_id );
    if ( ! $page || $page->post_type !== 'page' ) {
        wp_redirect( add_query_arg( array(
            'page' => 'html-import',
            'html_import_notice' => 'error_index_page_set',
        ), admin_url( 'admin.php' ) ) );
        exit;
    }

    if ( $page->post_status !== 'publish' ) {
        wp_update_post( array(
            'ID' => $page_id,
            'post_status' => 'publish',
        ) );
    }

    update_option( 'show_on_front', 'page' );
    update_option( 'page_on_front', $page_id );
    update_option( 'html_import_index_page_id', $page_id );

    html_import_configure_theme_front_page( $page_id );

    wp_redirect( add_query_arg( array(
        'page' => 'html-import',
        'html_import_notice' => 'success_index_page_set',
    ), admin_url( 'admin.php' ) ) );
    exit;
}
add_action( 'admin_post_html_import_set_index_page', 'html_import_handle_set_index_page' );

/**
 * テーマ固有のフロントページ設定を行います。
 */
function html_import_configure_theme_front_page( $page_id ) {
    $theme = wp_get_theme();
    $theme_slug = $theme->get_stylesheet();

    if ( strpos( $theme_slug, 'cocoon' ) !== false ) {
        update_option( 'front_page_type', 'page' );
        
        $cocoon_options = get_option( 'theme_mods_' . $theme_slug, array() );
        $cocoon_options['front_page_type'] = 'page';
        update_option( 'theme_mods_' . $theme_slug, $cocoon_options );
    }
}